/* 
 Name:
	Create3DFallingDominoes
 Version:
	1.6 (15 August 2010)
 Author:
	Charles Bordenave
 Description:  
	This script creates and animates a chain of 3D falling dominoes.
 Usage:
	In After Effects CS4 or later, run the script  
	Specify domino settings: a domino is represented by a 3D box 
	Specify final comp settings: the comp containing the animation
	Specify animation settings: number of dominoes in the chain, distance (offset) between two adjacent dominoes, number of frames before collision between two dominoes (this parameter controls the falling speed) 
	Click on Create to create the animation
 Acknowledgements:
	This script is inspired by Pete Everett's Plot3D demo (http://www.codeproject.com/KB/GDI-plus/Plot3D.aspx): some of his C++ functions have been converted to AE expression
*/


// This class represents the main class of the script
function Create3DFallingDominoes()
{
	// Variable used to keep track of 'this' reference
	var create3DFallingDominoes = this;
	
	// Create an instance of the utils class to use its functions
	var utils = new Create3DFallingDominoesUtils();

	// Script infos
	this.scriptMinSupportVersion = "9.0";
	this.scriptName = "Create3DFallingDominoes.jsx";	
	this.scriptVersion = "1.6";
	this.scriptTitle = "Create 3D Falling Dominoes";
	this.scriptCopyright = "Copyright (c) 2010 Charles Bordenave";
	this.scriptHomepage = "http://www.nabscripts.com";
	this.scriptDescription = {en: "This script creates and animates a chain of 3D falling dominoes.", fr:"Ce script crée et anime une chaîne de dominos 3D qui tombent."};
	this.scriptAbout = {en:this.scriptName + ", v" + this.scriptVersion + "\\r" + this.scriptCopyright + "\\r" + this.scriptHomepage + "\\r\\r" + utils.loc(this.scriptDescription), 
						fr:this.scriptName + ", v" + this.scriptVersion + "\\r" + this.scriptCopyright + "\\r" + this.scriptHomepage + "\\r\\r" + utils.loc(this.scriptDescription)};		
	this.scriptUsage = {en:	"\u25BA In After Effects CS4 or later, run the script \\r" +
							"\u25BA Specify domino settings: a domino is represented by a 3D box \\r" + 
							"\u25BA Specify final comp settings: the comp containing the animation \\r" +
							"\u25BA Specify animation settings: number of dominoes in the chain, distance (offset) between two adjacent dominoes, number of frames before collision between two dominoes (this parameter controls the falling speed) \\r" + 
							"\u25BA Click on Create to create the animation",
						fr:	"\u25BA Dans After Effects CS4 ou supérieur, exécuter le script \\r" +
							"\u25BA Spécifier les paramètres d\\\'un domino: un domino est représenté par une boîte 3D \\r" +
							"\u25BA Spécifier les paramètres de la comp finale: la comp contenant l\\\'animation \\r" +
							"\u25BA Spécifier les paramètres de l\\\'animation: nombre de dominos dans la chaîne, distance (décalage) entre les dominos, nombre d\\\'images avant la collision entre deux dominos (ce paramètre contrôle la vitesse de la chute) \\r" +
							"\u25BA Cliquer sur Créer pour créer l\\\'animation"};
	
	// Errors
	this.requirementErr = {en:"This script requires After Effects CS4 or later.", fr:"Ce script nécessite After Effects CS4 ou supérieur."};	
	this.noCompErr = {en:"A comp must be active.", fr:"Une composition doit être active."};
	this.noLayersErr = {en:"The active comp must contain at least one layer.", fr:"La composition active doit contenir au moins un calque."};

	// UI strings 
	this.aboutBtnName = "?";
	this.dominoPnlName = {en:"Domino Settings", fr:"Paramètres d\\'un domino"};
	this.dominoWidthHeightStName = {en:"Width/Height:", fr:"Largeur/Hauteur:"};
	this.dominoWidthEtDflt = 100;
	this.dominoHeightEtDflt = 140;
	this.dominoThicknessStName = {en:"Thickness:", fr:"Epaisseur:"};
	this.dominoThicknessEtDflt = 20;
	this.finalCompPnlName = {en:"Final Comp", fr:"Comp Finale"};
	this.finalCompWidthHeightStName = {en:"Width/Height:", fr:"Largeur/Hauteur:"};
	this.finalCompWidthEtDflt = 800;
	this.finalCompHeightEtDflt = 600;
	this.finalCompFrameRateStName = {en:"Frame Rate:", fr:"Cadence:"};
	this.finalCompFrameRateEtDflt = 25;
	this.finalCompDurationStName = {en:"Duration:", fr:"Durée:"};
	this.finalCompDurationEtDflt = 10;
	this.animationPnlName = {en:"Animation", fr:"Animation"};
	this.numberOfDominoesStName = {en:"Number Of Dominoes:", fr:"Nombre de dominos:"};	
	this.numberOfDominoesEtDflt = 10;
	this.distanceBetweenDominoesStName = {en:"Distance Between Dominoes:", fr:"Distance entre les dominos:"};
	this.distanceBetweenDominoesEtDflt = 100;
	this.framesBeforeCollisionStName = {en:"Frames Before Collision:", fr:"Nombre d\\'images avant collision:"};
	this.framesBeforeCollisionEtDflt = 5;
	this.runBtnName = {en:"Create", fr:"Créer"};
	
	// Domino-related strings & default settings
	this.dominoPrecompName = {en:"Domino Precomp", fr:"Précomp Domino"};
	this.sideLayerNames = {en:["Front","Back","Left","Right","Bottom","Top"], fr:["Avant","Arrière","Gauche","Droite","Bas","Haut"]};
	this.dominoColor = [1,1,1]; // white
	this.finalCompName = {en:"Domino Animation", fr:"Animation des dominos"};
	this.controllerLayerName = {en:"Controller", fr:"Contrôleur"};
	this.distanceBetweenDominoesEffectName = {en:"Distance Between Dominoes", fr:"Distance entre les dominos"};
	this.framesBeforeCollisionEffectName = {en:"Frames Before Collision", fr:"Nombre d'images avant collision"};
	this.firstDominoLayerName = {en:"First Domino", fr:"Premier Domino"};
	
	
	// Creates and displays the script interface
	this.buildUI = function (thisObj)
	{
		// dockable panel or palette
		var pal = (thisObj instanceof Panel) ? thisObj : new Window("palette", this.scriptTitle, undefined, {resizeable:false});

		// resource specifications
		var res =
		"group { orientation:'column', alignment:['left','top'], alignChildren:['right','top'], \
			gr1: Group { \
				aboutBtn: Button { text:'" + this.aboutBtnName + "', preferredSize:[25,20] } \
			}, \
			gr2: Panel { text:'" + utils.loc(this.dominoPnlName) + "', alignment:['fill','fill'], alignChildren:['right','top'], \
				gr21: Group { \
					dominoWidthHeightSt: StaticText { text:'" + utils.loc(this.dominoWidthHeightStName) + "' }, \
					dominoWidthEt: EditText { text:'" + this.dominoWidthEtDflt + "', characters:5 }, \
					dominoHeightEt: EditText { text:'" + this.dominoHeightEtDflt + "', characters:5 } \
				}, \
				gr22: Group { \
					dominoThicknessSt: StaticText { text:'" + utils.loc(this.dominoThicknessStName) + "' }, \
					dominoThicknessEt: EditText { text:'" + this.dominoThicknessEtDflt + "', characters:5 }, \
					fooEt: EditText { characters:5, visible:false } \
				} \
			}, \
			gr3: Panel { text:'" + utils.loc(this.finalCompPnlName) + "', alignment:['fill','fill'], alignChildren:['right','top'], \
				gr31: Group { \
					finalCompWidthHeightSt: StaticText { text:'" + utils.loc(this.finalCompWidthHeightStName) + "' }, \
					finalCompWidthEt: EditText { text:'" + this.finalCompWidthEtDflt + "', characters:5 }, \
					finalCompHeightEt: EditText { text:'" + this.finalCompHeightEtDflt + "', characters:5 } \
				}, \
				gr32: Group { \
					finalCompFrameRateSt: StaticText { text:'" + utils.loc(this.finalCompFrameRateStName) + "' }, \
					finalCompFrameRateEt: EditText { text:'" + this.finalCompFrameRateEtDflt + "', characters:5 }, \
					fooEt: EditText { characters:5, visible:false } \
				}, \
				gr33: Group { \
					finalCompDurationSt: StaticText { text:'" + utils.loc(this.finalCompDurationStName) + "' }, \
					finalCompDurationEt: EditText { text:'" + this.finalCompDurationEtDflt + "', characters:5 }, \
					fooEt: EditText { characters:5, visible:false } \
				} \
			}, \
			gr4: Panel { text:'" + utils.loc(this.animationPnlName) + "', alignment:['fill','fill'], alignChildren:['right','top'], \
				gr41: Group { \
					numberOfDominoesSt: StaticText { text:'" + utils.loc(this.numberOfDominoesStName) + "' }, \
					numberOfDominoesEt: EditText { text:'" + this.numberOfDominoesEtDflt + "', characters:5 }, \
					fooEt: EditText { characters:5, visible:false } \
				}, \
				gr42: Group { \
					distanceBetweenDominoesSt: StaticText { text:'" + utils.loc(this.distanceBetweenDominoesStName) + "' }, \
					distanceBetweenDominoesEt: EditText { text:'" + this.distanceBetweenDominoesEtDflt + "', characters:5 }, \
					fooEt: EditText { characters:5, visible:false } \
				}, \
				gr43: Group { \
					framesBeforeCollisionSt: StaticText { text:'" + utils.loc(this.framesBeforeCollisionStName) + "' }, \
					framesBeforeCollisionEt: EditText { text:'" + this.framesBeforeCollisionEtDflt + "', characters:5 }, \
					fooEt: EditText { characters:5, visible:false } \
				} \
			}, \
			gr5: Group { orientation:'row', alignment:['fill','top'], \
				runBtn: Button { text:'" + utils.loc(this.runBtnName) + "', alignment:['right','center'] } \
			} \
		}"; 
		pal.gr = pal.add(res);
		
		// event callbacks
		pal.gr.gr1.aboutBtn.onClick = function () 
		{ 
			utils.createAboutDlg(create3DFallingDominoes.scriptAbout, create3DFallingDominoes.scriptUsage); 
		};

		pal.gr.gr2.gr21.dominoWidthEt.onChange = function () 
		{ 
			if (isNaN(this.text) || parseInt(this.text) < 1 || parseInt(this.text) > 30000) this.text = create3DFallingDominoes.dominoWidthEtDflt;
			else this.text = Math.round(this.text);
		};		

		pal.gr.gr2.gr21.dominoHeightEt.onChange = function () 
		{ 
			if (isNaN(this.text) || parseInt(this.text) < 1 || parseInt(this.text) > 30000) this.text = create3DFallingDominoes.dominoHeightEtDflt;
			else this.text = Math.round(this.text);
		};

		pal.gr.gr2.gr22.dominoThicknessEt.onChange = function () 
		{ 
			if (isNaN(this.text) || parseInt(this.text) < 1 || parseInt(this.text) > 30000) this.text = create3DFallingDominoes.dominoThicknessEtDflt;
			else this.text = Math.round(this.text);
		};

		pal.gr.gr3.gr31.finalCompWidthEt.onChange = function () 
		{ 
			if (isNaN(this.text) || parseInt(this.text) < 1 || parseInt(this.text) > 30000) this.text = create3DFallingDominoes.finalCompWidthEtDflt;
			else this.text = Math.round(this.text);
		};		

		pal.gr.gr3.gr31.finalCompHeightEt.onChange = function () 
		{ 
			if (isNaN(this.text) || parseInt(this.text) < 1 || parseInt(this.text) > 30000) this.text = create3DFallingDominoes.finalCompHeightEtDflt;
			else this.text = Math.round(this.text);
		};

		pal.gr.gr3.gr32.finalCompFrameRateEt.onChange = function () 
		{ 
			if (isNaN(this.text) || parseInt(this.text) < 1 || parseInt(this.text) > 99) this.text = create3DFallingDominoes.finalCompFrameRateEtDflt;
			else this.text = Math.round(this.text);
		};
		
		pal.gr.gr3.gr33.finalCompDurationEt.onChange = function () 
		{ 
			if (isNaN(this.text) || parseFloat(this.text) <= 0 || parseFloat(this.text) > 10800) this.text = create3DFallingDominoes.finalCompDurationEtDflt;
			else this.text = parseFloat(this.text);
		};
						
		pal.gr.gr4.gr41.numberOfDominoesEt.onChange = function () 
		{ 
			if (isNaN(this.text) || parseInt(this.text) < 1 || parseInt(this.text) > 1000) this.text = create3DFallingDominoes.numberOfDominoesEtDflt;
			else this.text = Math.round(this.text);
		};

		pal.gr.gr4.gr42.distanceBetweenDominoesEt.onChange = function () 
		{ 
			if (isNaN(this.text) || parseFloat(this.text) <= 0 || parseFloat(this.text) >= pal.gr.gr2.gr21.dominoHeight) this.text = create3DFallingDominoes.distanceBetweenDominoesEtDflt;
			else this.text = parseFloat(this.text);
		};

		pal.gr.gr4.gr43.framesBeforeCollisionEt.onChange = function () 
		{ 
			if (isNaN(this.text) || parseInt(this.text) < 1 || parseInt(this.text) > 10800) this.text = create3DFallingDominoes.framesBeforeCollisionEtDflt;
			else this.text = Math.round(this.text);
		};
		
		pal.gr.gr5.runBtn.onClick = function ()
		{
			create3DFallingDominoes.createFallingDominoes(pal);	
		};
		
		// show user interface
		if (pal instanceof Window)
		{
			pal.center();
			pal.show();
		}
		else
		{
			pal.layout.layout(true);
		}	   
	};

	// Creates a domino precomp of given characteristics
	this.createDomino = function (pal)
	{
		var dominoWidth = parseInt(pal.gr.gr2.gr21.dominoWidthEt.text);
		var dominoHeight = parseInt(pal.gr.gr2.gr21.dominoHeightEt.text);
		var dominoThickness = parseInt(pal.gr.gr2.gr22.dominoThicknessEt.text);
						  
		var compName = utils.loc(this.dominoPrecompName);
		var compW = dominoWidth;					
		var compH = dominoHeight;					
		var compPAR = 1.0;
		var compDur = parseFloat(pal.gr.gr3.gr33.finalCompDurationEt.text);				   
		var compFPS = parseInt(pal.gr.gr3.gr32.finalCompFrameRateEt.text);				   

		var comp = app.project.items.addComp(compName, compW, compH, compPAR, compDur, compFPS);
		
		var pos = [ 
		[dominoWidth/2, dominoHeight/2, 0], 
		[dominoWidth/2, dominoHeight/2, dominoThickness],
		[0, dominoHeight/2, dominoThickness/2],
		[dominoWidth, dominoHeight/2, dominoThickness/2],
		[dominoWidth/2, dominoHeight, dominoThickness/2],
		[dominoWidth/2, 0, dominoThickness/2] 
		];
		var ori = [ 
		[0, 0, 0],		
		[0, 180, 0],
		[0, 90, 0],
		[0, 270, 0],
		[90, 0, 0],
		[270, 0, 0] 
		];
		var widths = [dominoWidth, dominoWidth, dominoThickness, dominoThickness, dominoWidth, dominoWidth];
		var heights = [dominoHeight, dominoHeight, dominoHeight, dominoHeight, dominoThickness, dominoThickness];
		
		for (var i = 0; i < 6; i++)
		{
			var layer = comp.layers.addSolid(this.dominoColor, utils.loc(this.sideLayerNames)[i], widths[i], heights[i], compPAR, compDur);
			layer.threeDLayer = true;
			layer.position.setValue(pos[i]);
			layer.orientation.setValue(ori[i]);
		}
				
		return comp;
	};

	// Creates and animates a chain of falling dominoes 
	this.createDominoesChain = function (pal, dominoPrecomp)
	{
		var dominoWidth = parseInt(pal.gr.gr2.gr21.dominoWidthEt.text);
		var dominoHeight = parseInt(pal.gr.gr2.gr21.dominoHeightEt.text);
		var dominoThickness = parseInt(pal.gr.gr2.gr22.dominoThicknessEt.text);
		var numberOfDominoes = parseInt(pal.gr.gr4.gr41.numberOfDominoesEt.text); 
		var distanceBetweenDominoes = parseFloat(pal.gr.gr4.gr42.distanceBetweenDominoesEt.text);
		var framesBeforeCollision = parseFloat(pal.gr.gr4.gr43.framesBeforeCollisionEt.text);
		
		var compName = utils.loc(this.finalCompName);
		var compW = parseInt(pal.gr.gr3.gr31.finalCompWidthEt.text);					
		var compH = parseInt(pal.gr.gr3.gr31.finalCompHeightEt.text);					
		var compPAR = 1.0;
		var compDur = parseFloat(pal.gr.gr3.gr33.finalCompDurationEt.text);				   
		var compFPS = parseInt(pal.gr.gr3.gr32.finalCompFrameRateEt.text);				   

		var comp = app.project.items.addComp(compName, compW, compH, compPAR, compDur, compFPS);	   
	
		// add controller
		var controllerLayer = comp.layers.addNull();
		controllerLayer.name = utils.loc(this.controllerLayerName);
		controllerLayer.threeDLayer = true;

		var halfZ = -(numberOfDominoes - 1) * (distanceBetweenDominoes + dominoThickness) / 2;
		controllerLayer.position.setValue([compW/2, compH/2, halfZ]);
		
		var distanceBetweenDominoesEffect = controllerLayer.Effects.addProperty("ADBE Slider Control");
		distanceBetweenDominoesEffect.name = utils.loc(this.distanceBetweenDominoesEffectName);
		distanceBetweenDominoesEffect.property(1).setValue(distanceBetweenDominoes);
		
		var framesBeforeCollisionEffect = controllerLayer.Effects.addProperty("ADBE Slider Control");
		framesBeforeCollisionEffect.name = utils.loc(this.framesBeforeCollisionEffectName);
		framesBeforeCollisionEffect.property(1).setValue(framesBeforeCollision);

		// add first domino
		var firstDomino = comp.layers.add(dominoPrecomp);
		firstDomino.name = utils.loc(this.firstDominoLayerName);
		firstDomino.threeDLayer = true;
		firstDomino.collapseTransformation = true;
		
		firstDomino.anchorPoint.setValue([dominoWidth/2, dominoHeight, 0]); 
		firstDomino.position.setValue([compW/2, compH/2 + dominoHeight/2, 0]);
		firstDomino.rotationX.expression = // this code is adapted from Pete Everett's Plot3D demo (http://www.codeproject.com/KB/GDI-plus/Plot3D.aspx) 
		"function solveQuadraticEquation(a, b, c)\r" +
		"{\r" +
		"	var solution1 = (-b + Math.sqrt(b * b - 4.0 * a * c)) / (2.0 * a);\r" +
		"	var solution2 = (-b - Math.sqrt(b * b - 4.0 * a * c)) / (2.0 * a);\r" +
		"	return Math.max(solution1, solution2);\r" +
		"}\r" +
		"\r" +
		"function findInnerAnglePhase1(bottomLength, dominoHeight, outerAngle)\r" +
		"{\r" +
		"	var tan = Math.tan(outerAngle);\r" +
		"	var a = (tan * tan) + 1;\r" +
		"	var b = 2 * bottomLength;\r" +
		"	var c = (bottomLength * bottomLength) - (dominoHeight * dominoHeight);\r" +
		"	var solution = solveQuadraticEquation(a, b, c);\r" +
		"	return Math.acos((bottomLength + solution) / dominoHeight);\r" +
		"}\r" +
		"\r" +
		"function getProjectedLength(tiltAngle, thickness)\r" +
		"{\r" +
		"	return thickness / Math.sin(tiltAngle);\r" +
		"}\r" +
		"\r" +
		"function findInnerAnglePhase2(outerAngle, thickness, distanceBetween)\r" +
		"{\r" +
		"	var y = thickness * Math.sin(Math.PI / 2 - outerAngle);\r" +
		"	var x = thickness * Math.cos(Math.PI / 2 - outerAngle);\r" +
		"	return Math.atan(y / (distanceBetween + thickness - x));\r" +		
		"}\r" +
		"\r" +
		"function findInnerAngle(outerAngle, dominoHeight, dominoThickness, distanceBetween)\r" +
		"{\r" +
		"	outerAngle = degreesToRadians(outerAngle);\r" +
		"	var projectedLength = getProjectedLength(outerAngle, dominoThickness);\r" +		
		"	var bottomLength = dominoThickness + distanceBetween - projectedLength;\r" +
		"	if (bottomLength > 0)\r" +
		"	{\r" +
		"		return radiansToDegrees(findInnerAnglePhase1(bottomLength, dominoHeight, outerAngle));\r" +			
		"	}\r" +
		"	else\r" +
		"	{\r" +
		"		return radiansToDegrees(findInnerAnglePhase2(outerAngle, dominoThickness, distanceBetween));\r" +
		"	}\r" +
		"}\r" +
		"\r" +
		"function Domino(height, thickness)\r" +
		"{\r" +
		"	var D = new Object();\r" +
		"	D.height = height;\r" +
		"	D.thickness = thickness;\r" +
		"	D.fallAngle = 90;\r" + 
		"	return D;\r" +
		"}\r" +
		"\r" +
		"function Dominoes(numberOfDominoes, dominoHeight, dominoThickness, distanceBetweenDominoes, framesBeforeCollision)\r" +
		"{\r" +
		"	var Ds = new Object();\r" +
		"	Ds.dominoArray = new Array(numberOfDominoes);\r" +
		"	for (var d = 0; d < Ds.dominoArray.length; d++)\r" +
		"	{\r" +
		"		Ds.dominoArray[d] = new Domino(dominoHeight, dominoThickness);\r" +
		"	}\r" +
		"	Ds.distanceBetweenDominoes = distanceBetweenDominoes;\r" +
		"	Ds.angleBetweenDominoes = radiansToDegrees(Math.acos(Ds.distanceBetweenDominoes / dominoHeight));\r" +
		"\r" +
		"	Ds.calculateRotation = function ()\r" +
		"	{\r" +		 
		"		for (var lastFallingDomino = 0; lastFallingDomino < Ds.dominoArray.length; lastFallingDomino++)\r" +
		"		{\r" +
		"			for (var frame = 0; frame < framesBeforeCollision; frame++)\r" +
		"			{\r" +
		"				Ds.dominoArray[lastFallingDomino].fallAngle = 90 - ((90 - Ds.angleBetweenDominoes) * frame / framesBeforeCollision);\r" +
		"\r" +
		"				for (var parentDomino = lastFallingDomino - 1; parentDomino >= 0; parentDomino--)\r" +
		"				{\r" +
		"					Ds.dominoArray[parentDomino].fallAngle = findInnerAngle(Ds.dominoArray[parentDomino + 1].fallAngle,\r" +
		"																			Ds.dominoArray[0].height,\r" + 
		"																			Ds.dominoArray[0].thickness,\r" +
		"																			Ds.distanceBetweenDominoes);\r" +
		"\r" +
		"					if (parentDomino == 0)\r" +
		"					{\r" +
		"						rotationArray[rotationArray.length] = Ds.dominoArray[parentDomino].fallAngle;\r" +
		"					}\r" +
		"				}\r" +
		"			}\r" +
		"		}\r" +
		"\r" +
		"		var lastDomino = Ds.dominoArray[Ds.dominoArray.length - 1];\r" +
		"\r" +
		"		var fallAngle = lastDomino.fallAngle;\r" +
		"\r" +
		"		var distancePerFrame = Ds.angleBetweenDominoes / framesBeforeCollision;\r" +
		"\r" +
		"		while (fallAngle > 0)\r" +
		"		{\r" +
		"			fallAngle -= distancePerFrame;\r" +
		"\r" +
		"			lastDomino.fallAngle = Math.max(fallAngle, 0);\r" +
		"\r" +
		"			for (var parentDomino = Ds.dominoArray.length - 2; parentDomino >= 0; parentDomino--)\r" +
		"			{\r" +
		"				Ds.dominoArray[parentDomino].fallAngle = findInnerAngle(Ds.dominoArray[parentDomino + 1].fallAngle,\r" + 
		"																		Ds.dominoArray[0].height,\r" + 
		"																		Ds.dominoArray[0].thickness,\r" + 
		"																		Ds.distanceBetweenDominoes);\r" +
		"\r" +
		"				if (parentDomino == 0)\r" +
		"				{\r" +
		"					rotationArray[rotationArray.length] = Ds.dominoArray[parentDomino].fallAngle;\r" +
		"				}\r" +
		"			}\r" +
		"		}\r" +		
		"	};\r" +
		"\r" +
		"	return Ds;\r" +
		"}\r" +
		"\r" +
		"numberOfDominoes = " + numberOfDominoes + ";\r" +
		"dominoHeight = height;\r" +
		"dominoThickness = " + dominoThickness + ";\r" +
		"distanceBetweenDominoes = thisComp.layer(\"" + utils.loc(this.controllerLayerName) + "\").effect(\"" + utils.loc(this.distanceBetweenDominoesEffectName) + "\")(1);\r" +
		"framesBeforeCollision = thisComp.layer(\"" + utils.loc(this.controllerLayerName) + "\").effect(\"" + utils.loc(this.framesBeforeCollisionEffectName) + "\")(1);\r" +
		"Ds = new Dominoes(numberOfDominoes, dominoHeight, dominoThickness, distanceBetweenDominoes, framesBeforeCollision);\r" +
		"\r" +
		"rotationArray = new Array();\r" +		
		"Ds.calculateRotation();\r" +
		"\r" +
		"lastAngle = radiansToDegrees(Math.PI/2 - Math.asin(dominoThickness / (distanceBetweenDominoes + dominoThickness)));\r" +
		"f = timeToFrames(time);\r" +
		"\r" +
		"if (f <= framesBeforeCollision)\r" + 
		"{\r" +
		"   linear(time,0,framesBeforeCollision * thisComp.frameDuration,0,90 - rotationArray[0]);\r" +
		"}\r" +
		"else if (f < Ds.dominoArray.length * framesBeforeCollision)\r" +
		"{\r" +
		"   90 - rotationArray[f - framesBeforeCollision];\r" +
		"}\r" +
		"else\r" +
		"{\r" +
		"   lastAngle;\r" +
		"}";
		
		firstDomino.parent = controllerLayer;
		firstDomino.moveAfter(comp.layer(comp.numLayers));
		
		// add other dominoes
	   for (var d = 1; d < numberOfDominoes; d++)
		{
			var curDomino = comp.layers.add(dominoPrecomp);
			curDomino.threeDLayer = true;
			curDomino.collapseTransformation = true;
			curDomino.anchorPoint.setValue([dominoPrecomp.width / 2, dominoPrecomp.height, 0]);						
			//curDomino.position.setValue([comp.width / 2, comp.height / 2 + dominoComp.height / 2, -d * (G.ANIM_DISTANCE_BETWEEN + G.DOMINO_THICKNESS)]);
			
			curDomino.moveAfter(comp.layer(comp.numLayers)); // before applying expression since the expression uses index-1
			
			curDomino.position.expression = 
			"distanceBetweenDominoes = thisComp.layer(\"" + utils.loc(this.controllerLayerName) + "\").effect(\"" + utils.loc(this.distanceBetweenDominoesEffectName) + "\")(1);\r" +
			"dominoThickness = " + dominoThickness + ";\r" +
			"thisComp.layer(index - 1).position - [0,0,distanceBetweenDominoes + dominoThickness];";			
			
			curDomino.rotationX.expression =
			"framesBeforeCollision = thisComp.layer(\"" + utils.loc(this.controllerLayerName) + "\").effect(\"" + utils.loc(this.framesBeforeCollisionEffectName) + "\")(1);\r" +
			"thisComp.layer(index - 1).rotationX.valueAtTime(time - framesBeforeCollision * thisComp.frameDuration);";					
			
			curDomino.parent = controllerLayer;
			curDomino.selected = false;
		}			
		
		// something like Left View 
		controllerLayer.position.setValue([compW/2, compH/2, -halfZ]);
		controllerLayer.rotationY.setValue(-90); 
	};
	
	// Creates a domino precomp and use it to build and animate a chain of falling dominoes
	this.createFallingDominoes = function (pal)
	{
		try
		{
			app.beginUndoGroup(this.scriptTitle);
			
			var dominoWidth = parseInt(pal.gr.gr2.gr21.dominoWidthEt.text);
			var dominoHeight = parseInt(pal.gr.gr2.gr21.dominoHeightEt.text);
			var dominoThickness = parseInt(pal.gr.gr2.gr22.dominoThicknessEt.text);
			
			var dominoPrecomp = this.createDomino(pal);
			
			this.createDominoesChain(pal, dominoPrecomp);
			  
			app.endUndoGroup();
		}
		catch(e)//rr)
		{
			alert(e);
			//utils.throwErr(err);
		}				
	};
	
	// Runs the script  
	this.run = function (thisObj) 
	{
		if (parseFloat(app.version) < parseFloat(this.scriptMinSupportVersion))
		{
			utils.throwErr(this.requirementErr);
		}
		else
		{
			this.buildUI(thisObj);
		}	
	};
}


// This class provides some utility functions
function Create3DFallingDominoesUtils()
{
	// Variable used to keep track of 'this' reference
	var utils = this;
	
	// String localization function: english and french languages are supported
	this.loc = function (str)
	{
		return app.language == Language.FRENCH ? str.fr : str.en;
	};

	// Displays a window containg a localized error message
	this.throwErr = function (err)
	{
		var wndTitle = $.fileName.substring($.fileName.lastIndexOf("/")+1, $.fileName.lastIndexOf("."));
		Window.alert("Script error:\r" + this.loc(err), wndTitle, true);
	};			

	// Displays a dialog containg the script description and usage
	this.createAboutDlg = function (aboutStr, usageStr)
	{	
		eval(unescape('%09%09%76%61%72%20%64%6C%67%20%3D%20%6E%65%77%20%57%69%6E%64%6F%77%28%22%64%69%61%6C%6F%67%22%2C%20%22%41%62%6F%75%74%22%29%3B%0A%09%20%20%20%20%20%20%09%20%20%20%20%20%20%20%09%0A%09%20%20%20%20%76%61%72%20%72%65%73%20%3D%0A%09%09%22%67%72%6F%75%70%20%7B%20%6F%72%69%65%6E%74%61%74%69%6F%6E%3A%27%63%6F%6C%75%6D%6E%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%66%69%6C%6C%27%2C%27%66%69%6C%6C%27%5D%2C%20%61%6C%69%67%6E%43%68%69%6C%64%72%65%6E%3A%5B%27%66%69%6C%6C%27%2C%27%66%69%6C%6C%27%5D%2C%20%5C%0A%09%09%09%70%6E%6C%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%62%65%64%70%61%6E%65%6C%27%2C%20%5C%0A%09%09%09%09%61%62%6F%75%74%54%61%62%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%27%2C%20%74%65%78%74%3A%27%44%65%73%63%72%69%70%74%69%6F%6E%27%2C%20%5C%0A%09%09%09%09%09%61%62%6F%75%74%45%74%3A%20%45%64%69%74%54%65%78%74%20%7B%20%74%65%78%74%3A%27%22%20%2B%20%74%68%69%73%2E%6C%6F%63%28%61%62%6F%75%74%53%74%72%29%20%2B%20%22%27%2C%20%70%72%65%66%65%72%72%65%64%53%69%7A%65%3A%5B%33%36%30%2C%32%30%30%5D%2C%20%70%72%6F%70%65%72%74%69%65%73%3A%7B%6D%75%6C%74%69%6C%69%6E%65%3A%74%72%75%65%7D%20%7D%20%5C%0A%09%09%09%09%7D%2C%20%5C%0A%09%09%09%09%75%73%61%67%65%54%61%62%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%27%2C%20%74%65%78%74%3A%27%55%73%61%67%65%27%2C%20%5C%0A%09%09%09%09%09%75%73%61%67%65%45%74%3A%20%45%64%69%74%54%65%78%74%20%7B%20%74%65%78%74%3A%27%22%20%2B%20%74%68%69%73%2E%6C%6F%63%28%75%73%61%67%65%53%74%72%29%20%2B%20%22%27%2C%20%70%72%65%66%65%72%72%65%64%53%69%7A%65%3A%5B%33%36%30%2C%32%30%30%5D%2C%20%70%72%6F%70%65%72%74%69%65%73%3A%7B%6D%75%6C%74%69%6C%69%6E%65%3A%74%72%75%65%7D%20%7D%20%5C%0A%09%09%09%09%7D%20%5C%0A%09%09%09%7D%2C%20%5C%0A%09%09%09%62%74%6E%73%3A%20%47%72%6F%75%70%20%7B%20%6F%72%69%65%6E%74%61%74%69%6F%6E%3A%27%72%6F%77%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%66%69%6C%6C%27%2C%27%62%6F%74%74%6F%6D%27%5D%2C%20%5C%0A%09%09%09%09%6F%74%68%65%72%53%63%72%69%70%74%73%42%74%6E%3A%20%42%75%74%74%6F%6E%20%7B%20%74%65%78%74%3A%27%4F%74%68%65%72%20%53%63%72%69%70%74%73%2E%2E%2E%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%6C%65%66%74%27%2C%27%63%65%6E%74%65%72%27%5D%20%7D%2C%20%5C%0A%09%09%09%09%6F%6B%42%74%6E%3A%20%42%75%74%74%6F%6E%20%7B%20%74%65%78%74%3A%27%4F%6B%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%72%69%67%68%74%27%2C%27%63%65%6E%74%65%72%27%5D%20%7D%20%5C%0A%09%09%09%7D%20%5C%0A%09%09%7D%22%3B%20%0A%09%09%64%6C%67%2E%67%72%20%3D%20%64%6C%67%2E%61%64%64%28%72%65%73%29%3B%0A%09%09%0A%09%09%64%6C%67%2E%67%72%2E%70%6E%6C%2E%61%62%6F%75%74%54%61%62%2E%61%62%6F%75%74%45%74%2E%6F%6E%43%68%61%6E%67%65%20%3D%20%64%6C%67%2E%67%72%2E%70%6E%6C%2E%61%62%6F%75%74%54%61%62%2E%61%62%6F%75%74%45%74%2E%6F%6E%43%68%61%6E%67%69%6E%67%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%74%68%69%73%2E%74%65%78%74%20%3D%20%75%74%69%6C%73%2E%6C%6F%63%28%61%62%6F%75%74%53%74%72%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%72%2F%67%2C%20%27%5C%72%27%29%3B%0A%09%09%7D%3B%0A%09%09%0A%09%09%64%6C%67%2E%67%72%2E%70%6E%6C%2E%75%73%61%67%65%54%61%62%2E%75%73%61%67%65%45%74%2E%6F%6E%43%68%61%6E%67%65%20%3D%20%64%6C%67%2E%67%72%2E%70%6E%6C%2E%75%73%61%67%65%54%61%62%2E%75%73%61%67%65%45%74%2E%6F%6E%43%68%61%6E%67%69%6E%67%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%74%68%69%73%2E%74%65%78%74%20%3D%20%75%74%69%6C%73%2E%6C%6F%63%28%75%73%61%67%65%53%74%72%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%72%2F%67%2C%20%27%5C%72%27%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%27%2F%67%2C%20%22%27%22%29%3B%0A%09%09%7D%3B%0A%09%09%09%0A%09%09%64%6C%67%2E%67%72%2E%62%74%6E%73%2E%6F%74%68%65%72%53%63%72%69%70%74%73%42%74%6E%2E%6F%6E%43%6C%69%63%6B%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%76%61%72%20%63%6D%64%20%3D%20%22%22%3B%0A%09%09%09%76%61%72%20%75%72%6C%20%3D%20%22%68%74%74%70%3A%2F%2F%61%65%73%63%72%69%70%74%73%2E%63%6F%6D%2F%63%61%74%65%67%6F%72%79%2F%73%63%72%69%70%74%73%2F%6E%61%62%2F%22%3B%0A%09%0A%09%09%09%69%66%20%28%24%2E%6F%73%2E%69%6E%64%65%78%4F%66%28%22%57%69%6E%22%29%20%21%3D%20%2D%31%29%0A%09%09%09%7B%0A%09%20%20%20%20%20%20%20%20%09%69%66%20%28%46%69%6C%65%28%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%22%29%2E%65%78%69%73%74%73%29%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%09%65%6C%73%65%20%69%66%20%28%46%69%6C%65%28%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%20%28%78%38%36%29%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%22%29%2E%65%78%69%73%74%73%29%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%20%28%78%38%36%29%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%09%65%6C%73%65%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%49%6E%74%65%72%6E%65%74%20%45%78%70%6C%6F%72%65%72%2F%69%65%78%70%6C%6F%72%65%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%7D%0A%09%09%09%65%6C%73%65%0A%09%09%09%09%63%6D%64%20%2B%3D%20%22%6F%70%65%6E%20%5C%22%22%20%2B%20%75%72%6C%20%2B%20%22%5C%22%22%3B%20%20%20%20%20%20%20%20%20%0A%09%09%09%74%72%79%0A%09%09%09%7B%0A%09%09%09%09%73%79%73%74%65%6D%2E%63%61%6C%6C%53%79%73%74%65%6D%28%63%6D%64%29%3B%0A%09%09%09%7D%0A%09%09%09%63%61%74%63%68%28%65%29%0A%09%09%09%7B%0A%09%09%09%09%61%6C%65%72%74%28%65%29%3B%0A%09%09%09%7D%0A%09%09%7D%3B%0A%09%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%0A%09%09%64%6C%67%2E%67%72%2E%62%74%6E%73%2E%6F%6B%42%74%6E%2E%6F%6E%43%6C%69%63%6B%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%20%0A%09%09%7B%0A%09%09%09%64%6C%67%2E%63%6C%6F%73%65%28%29%3B%20%0A%09%09%7D%3B%0A%09%20%20%20%20%20%20%20%0A%09%09%64%6C%67%2E%63%65%6E%74%65%72%28%29%3B%0A%09%09%64%6C%67%2E%73%68%6F%77%28%29%3B'));		
	};
}


// Creates an instance of the main class and run it
new Create3DFallingDominoes().run(this);
